<?php

declare(strict_types=1);

namespace RhysLeesLtd\LaravelDefault\Commands;

use Artesaos\SEOTools\Providers\SEOToolsServiceProvider;
use Illuminate\Console\Command;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Facades\Process;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\MediaLibraryServiceProvider;

class InstallCommand extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $signature = 'default:install';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install the Laravel Default package';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        // Publish config
        $this->call('vendor:publish', ['--tag' => 'default-config', '--force' => true]);

        // Install CSP
        $this->csp();

        // Install Health
        $this->installHealth();

        // Install Media Library
        $this->installMediaLibrary();

        // Install Seo Tools
        $this->call('vendor:publish', ['--provider' => SEOToolsServiceProvider::class, '--force' => true]);

        // Install Schedule
        $this->installSchedule();

        // Install Rector
        $this->installRector();

        // Set env vars
        $this->setEnvVars();

        // Installation complete
        $this->components->info('✅ Laravel Default installation complete!');
        $this->components->twoColumnDetail('Next steps:', '');
        $this->components->bulletList([
            'Add <x-default::meta path="/path/to/favicons" /> to your layout head',
            'Add <x-default::userback /> to your layout body (optional)',
            'Configure environment variables in .env',
            'Run php artisan optimize after deployment',
        ]);

        // Add new files to git
        Process::run('git add .');
    }

    // Helpers
    protected function replaceInFile($search, $replace, $path)
    {
        file_put_contents($path, str_replace($search, $replace, file_get_contents($path)));
    }

    protected function appendInFile($search, $append, $path)
    {
        if (! Str::contains(file_get_contents($path), $search)) {
            (new Filesystem)->append($path, $append);
        }
    }

    protected function installSchedule()
    {
        $this->replaceInFile(
            search: <<<'EOT'
    ->withMiddleware(function (Middleware $middleware) {
        //
    })
EOT,
            replace: <<<'EOT'
    ->withMiddleware(function (Middleware $middleware) {
        //
    })
    ->withSchedule(function (Schedule $schedule) {
        LaravelDefault::scheduleDefaultCommands($schedule);

        $schedule->command('media-library:regenerate --only-missing --force')
            ->everyFifteenMinutes();
    })
EOT
            ,
            path: base_path('bootstrap/app.php')
        );

        $this->replaceInFile(
            search: <<<'EOT'
use Illuminate\Foundation\Configuration\Middleware;
EOT,
            replace: <<<'EOT'
use Illuminate\Foundation\Configuration\Middleware;
use Illuminate\Console\Scheduling\Schedule;
use RhysLeesLtd\LaravelDefault\Facades\LaravelDefault;
EOT,
            path: base_path('bootstrap/app.php')
        );
    }

    // Health
    protected function installHealth(): void
    {
        $this->call('vendor:publish', ['--tag' => 'health-config', '--force' => true]);
        copy(__DIR__.'/../../stubs/app/Providers/HealthServiceProvider.php', app_path('Providers/HealthServiceProvider.php'));

        $this->replaceInFile(
            search: <<<'EOT'
    'oh_dear_endpoint' => [
        'enabled' => false,
EOT,
            replace: <<<'EOT'
    'oh_dear_endpoint' => [
        'enabled' => env('OH_DEAR_ENABLED', false),
EOT
            ,
            path: config_path('health.php')
        );

        $this->replaceInFile(
            search: <<<'EOT'
    'notifications' => [
        /*
         * Notifications will only get sent if this option is set to `true`.
         */
        'enabled' => true,
EOT,
            replace: <<<'EOT'
    'notifications' => [
        /*
         * Notifications will only get sent if this option is set to `true`.
         */
        'enabled' => false,
EOT
            ,
            path: config_path('health.php')
        );

        $this->replaceInFile(
            search: "'theme' => 'light',",
            replace: "'theme' => 'dark',",
            path: config_path('health.php')
        );
    }

    // CSP
    protected function csp(): void
    {
        $this->components->info('Installing CSP with modular presets...');
        
        $this->call('vendor:publish', ['--tag' => 'csp-config', '--force' => true]);

        $this->replaceInFile(
            search: 'Spatie\Csp\Presets\Basic::class,',
            replace: 'RhysLeesLtd\LaravelDefault\Policies\SecurityHeader::class,',
            path: config_path('csp.php')
        );

        $this->components->info('CSP configured with modular presets:');
        $this->components->bulletList([
            'Basic CSP directives',
            'Fathom Analytics',
            'Google Fonts',
            'DigitalOcean Spaces (conditional)',
            'Gravatar avatars',
            'UI Avatars',
            'Userback feedback widget',
            'Data/Blob URIs and WebSockets',
            'Development localhost (local/testing only)',
        ]);
    }

    private function installMediaLibrary(): void
    {
        $this->call('vendor:publish', [
            '--provider' => MediaLibraryServiceProvider::class,
            '--tag' => 'medialibrary-migrations',
            '--force' => true,
        ]);

        copy(__DIR__.'/../../stubs/config/media-library.php', config_path('media-library.php'));

        $this->replaceInFile(
            search: <<<'EOT'
    'default' => env('FILESYSTEM_DISK', 'local'),
EOT,
            replace: <<<'EOT'
    'default' => env('FILESYSTEM_DISK', 'local'),

    'use_disk' => [
        'public' => env('USE_DISK_PUBLIC', 'do_public'),
        'private' => env('USE_DISK_PRIVATE', 'do'),
    ],
EOT,
            path: config_path('filesystems.php')
        );

        $this->replaceInFile(
            search: <<<'EOT'
    's3' => [
            'driver' => 's3',
            'key' => env('AWS_ACCESS_KEY_ID'),
            'secret' => env('AWS_SECRET_ACCESS_KEY'),
            'region' => env('AWS_DEFAULT_REGION'),
            'bucket' => env('AWS_BUCKET'),
            'url' => env('AWS_URL'),
            'endpoint' => env('AWS_ENDPOINT'),
            'use_path_style_endpoint' => env('AWS_USE_PATH_STYLE_ENDPOINT', false),
            'throw' => false,
        ],
EOT,
            replace: <<<'EOT'
    'do' => [
            'driver' => 's3',
            'key' => env('DO_ACCESS_KEY_ID'),
            'secret' => env('DO_SECRET_ACCESS_KEY'),
            'region' => env('DO_DEFAULT_REGION', 'lon1'),
            'bucket' => env('DO_BUCKET'),
            'root' => env('APP_ENV', 'unknown'),
            'url' => env('DO_SUBDOMAIN_ENDPOINT', 'https://'.env('DO_BUCKET').'.'.env('DO_CDN_ENDPOINT', 'lon1.cdn.digitaloceanspaces.com')),
            'endpoint' => 'https://'.env('DO_ENDPOINT', 'lon1.digitaloceanspaces.com'),
            'cdn_endpoint' => 'https://'.env('DO_CDN_ENDPOINT', 'lon1.cdn.digitaloceanspaces.com'),
            'use_path_style_endpoint' => env('DO_USE_PATH_STYLE_ENDPOINT', false),
            'visibility' => 'private',
            'throw' => true,
        ],

        'do_public' => [
            'driver' => 's3',
            'key' => env('DO_ACCESS_KEY_ID'),
            'secret' => env('DO_SECRET_ACCESS_KEY'),
            'region' => env('DO_DEFAULT_REGION', 'lon1'),
            'bucket' => env('DO_BUCKET'),
            'root' => env('APP_ENV', 'unknown'),
            'url' => env('DO_SUBDOMAIN_ENDPOINT', 'https://'.env('DO_BUCKET').'.'.env('DO_CDN_ENDPOINT', 'lon1.cdn.digitaloceanspaces.com')),
            'endpoint' => 'https://'.env('DO_ENDPOINT', 'lon1.digitaloceanspaces.com'),
            'cdn_endpoint' => 'https://'.env('DO_CDN_ENDPOINT', 'lon1.cdn.digitaloceanspaces.com'),
            'use_path_style_endpoint' => env('DO_USE_PATH_STYLE_ENDPOINT', false),
            'visibility' => 'public',
            'throw' => true,
        ],
EOT,
            path: config_path('filesystems.php')
        );
    }

    public function setEnvVars(): void
    {
        $files = [
            '.env',
            '.env.example',
        ];

        $vars = [
            'DO_ACCESS_KEY_ID=',
            'DO_SECRET_ACCESS_KEY=',
            'DO_DEFAULT_REGION=lon1',
            'DO_BUCKET=example-test',
            'DO_USE_PATH_STYLE_ENDPOINT=false',
            'DO_SUBDOMAIN_ENDPOINT=https://content.example.test',
            '',
            'FATHOM_URL=https://cdn-eu.usefathom.com/script.js',
            'FATHOM_SITE_ID=',
            '',
            'FLARE_ENABLED=false',
            'FLARE_PROJECT_ID=',
            'FLARE_KEY=',
            '',
            'OH_DEAR_ENABLED=false',
            'OH_DEAR_SITE_ID=',
            'OH_DEAR_KEY=',
            'OH_DEAR_HEALTH_CHECK_SECRET=',
            '',
            'USERBACK_ENABLED=false',
            'USERBACK_ACCESS_TOKEN=',
            '',
            'CSP_ENABLED=true',
            'CSP_NONCE_ENABLED=true',
            'CSP_REPORT_URI=',
            '',
            'HONEYPOT_ENABLED=false',
            'DEBUGBAR_ENABLED=false',
        ];
        foreach ($files as $file) {
            foreach ($vars as $var) {
                $this->appendInFile(
                    search: Str::before($var, '='),
                    append: PHP_EOL.$var,
                    path: base_path($file)
                );

                $this->replaceInFile(
                    search: 'FILESYSTEM_DISK=local',
                    replace: <<<'EOT'
USE_DISK_PRIVATE=do
USE_DISK_PUBLIC=do_public

FILESYSTEM_DISK=do_public
EOT,
                    path: base_path($file)
                );
            }
        }
    }

    public function installRector(): void
    {
        $this->components->info('Installing Rector...');
        Process::run('composer require rector/rector --dev');
        $this->components->info('Installing Rector Laravel...');
        Process::run('composer require driftingly/rector-laravel --dev');
        $this->components->info('Publishing Rector Laravel Config...');
        Process::run('php artisan vendor:publish --provider="Rector\Rector\Rectors\RectorServiceProvider" --tag="config"');

        $this->components->info('Publishing Rector Config...');
        copy(__DIR__.'/../../stubs/rector.php', base_path('rector.php'));
    }
}
