/**
 * Barcode scanner (~...*) capture.
 * Listens on document for prefix ~ and suffix *, buffers keydowns in between,
 * then calls onBarcodeScanned(barcode). No input element required.
 *
 * @param {(barcode: string) => void} onBarcodeScanned - Callback when a full ~...* scan is received
 */

export function initBarcodeScanner(onBarcodeScanned, prefix = '~', suffix = '*') {
    const SCANNER_PREFIX = prefix;
    const SCANNER_SUFFIX = suffix;
    let barcodeBuffer = '';
    let lastInputTime = 0;
    let isCapturing = false;
    const INPUT_TIMEOUT = 100;

    document.addEventListener(
        'keydown',
        (e) => {
            const isPrefix =
                e.key === SCANNER_PREFIX ||
                (e.shiftKey && e.key === '`') ||
                e.keyCode === 192;
            if (isPrefix) {
                barcodeBuffer = SCANNER_PREFIX;
                isCapturing = true;
                lastInputTime = Date.now();
                e.preventDefault();
                e.stopPropagation();
                return;
            }
            if (!isCapturing) return;

            const now = Date.now();
            if (now - lastInputTime > INPUT_TIMEOUT) {
                barcodeBuffer = '';
                isCapturing = false;
                return;
            }
            lastInputTime = now;

            if (e.key === SCANNER_SUFFIX) {
                barcodeBuffer += SCANNER_SUFFIX;
                if (
                    barcodeBuffer.startsWith(SCANNER_PREFIX) &&
                    barcodeBuffer.endsWith(SCANNER_SUFFIX)
                ) {
                    const barcode = barcodeBuffer.slice(1, -1).trim();
                    onBarcodeScanned(barcode);
                    e.preventDefault();
                    e.stopPropagation();
                }
                barcodeBuffer = '';
                isCapturing = false;
                return;
            }

            if (e.key.length === 1 && !e.ctrlKey && !e.metaKey && !e.altKey) {
                barcodeBuffer += e.key;
                e.preventDefault();
                e.stopPropagation();
            }
        },
        true
    );
}
