/**
 * Barcode scanner (~...*) capture – Alpine.js component.
 * Listens on document for prefix ~ and suffix *, buffers keydowns in between,
 * then dispatches 'barcode-scanned' with detail { barcode }.
 * Use with x-data="barcodeScanner(prefix, suffix)" and @barcode-scanned="...".
 */

export default function barcodeScannerAlpinePlugin(Alpine) {
    Alpine.data('barcodeScanner', (prefix = '~', suffix = '*') => ({
        prefix,
        suffix,
        barcodeBuffer: '',
        lastInputTime: 0,
        isCapturing: false,
        INPUT_TIMEOUT: 100,

        init() {
            const handleKeydown = this.handleKeydown.bind(this);
            document.addEventListener('keydown', handleKeydown, true);

            if (typeof this.$cleanup === 'function') {
                this.$cleanup(() => {
                    document.removeEventListener('keydown', handleKeydown, true);
                });
            }
        },

        handleKeydown(e) {
            const consume = () => {
                e.preventDefault();
                e.stopPropagation();
            };
            const reset = () => {
                this.barcodeBuffer = '';
                this.isCapturing = false;
            };
            const isPrefix =
                e.key === this.prefix ||
                (e.shiftKey && e.key === '`') ||
                e.keyCode === 192;

            if (isPrefix) {
                this.barcodeBuffer = this.prefix;
                this.isCapturing = true;
                this.lastInputTime = Date.now();
                consume();
                return;
            }
            if (!this.isCapturing) return;

            const now = Date.now();
            if (now - this.lastInputTime > this.INPUT_TIMEOUT) {
                reset();
                return;
            }
            this.lastInputTime = now;

            if (e.key === this.suffix) {
                this.barcodeBuffer += this.suffix;
                const valid =
                    this.barcodeBuffer.startsWith(this.prefix) &&
                    this.barcodeBuffer.endsWith(this.suffix);
                if (valid) {
                    this.$el.dispatchEvent(
                        new CustomEvent('barcode-scanned', {
                            detail: { barcode: this.barcodeBuffer.slice(1, -1).trim() },
                            bubbles: true,
                        })
                    );
                }
                consume();
                reset();
                return;
            }

            if (e.key.length === 1 && !e.ctrlKey && !e.metaKey && !e.altKey) {
                this.barcodeBuffer += e.key;
                consume();
            }
        },
    }));
}

// Self-register when loaded as a script (e.g. by Filament)
if (typeof window !== 'undefined' && window.Alpine) {
    window.Alpine.plugin(barcodeScannerAlpinePlugin);
} else {
    document.addEventListener('alpine:init', () => {
        window.Alpine.plugin(barcodeScannerAlpinePlugin);
    });
}
