<?php

namespace Andreia\FilamentBusinessHours\Forms\Components;

use Andreia\FilamentBusinessHours\Enums\Days;
use Closure;
use Exception;
use Filament\Actions\Action;
use Filament\Forms\Components\Field;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Repeater\TableColumn;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\TimePicker;
use Filament\Forms\Components\Toggle;
use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Group;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use Tapp\FilamentTimezoneField\Forms\Components\TimezoneSelect;

class BusinessHoursField extends Field
{
    protected string $view = 'filament-business-hours::forms.components.business-hours';

    protected bool | Closure $allowExceptions = false;

    protected bool | Closure $disabledByDefault = false;

    protected function setUp(): void
    {
        parent::setUp();

        $this->dehydrated(false);

        $this->hiddenLabel();

        $hoursSchema = [];

        // Add day fields dynamically
        foreach (static::getDays() as $day) {
            $hoursSchema[] = static::getDaySchema($day);
        }

        $this->schema([
            Toggle::make('enabled')
                ->label(__('filament-business-hours::business-hours.form.enable.label'))
                ->helperText(__('filament-business-hours::business-hours.form.enable.helper_text')),
            TimezoneSelect::make('timezone')
                ->label(__('filament-business-hours::business-hours.form.timezone.label'))
                ->native(false)
                ->searchable()
                ->preload()
                ->language(App::currentLocale()),
            ...$hoursSchema,
            Repeater::make('exceptions')
                ->hiddenLabel()
                ->addable(false)
                ->defaultItems(0)
                ->extraAttributes(['class' => 'ring-0 business-hours-table-repeater'])
                ->extraFieldWrapperAttributes(['class' => 'ring-0 business-hours-table-repeater'])
                ->reorderable(false)
                ->table([
                    TableColumn::make('exception_placeholder')
                        ->hiddenHeaderLabel()
                        ->width('150px'),
                ])
                ->schema([
                    TextEntry::make('exception_placeholder')
                        ->grow()
                        ->hiddenLabel()
                        ->html()
                        ->formatStateUsing(function (Get $get, $state) {
                            $exceptionData = $state;

                            if (is_string($exceptionData)) {
                                $data = json_decode($exceptionData, true);
                            } else {
                                $data = $exceptionData;
                            }

                            if (! $data) {
                                return '';
                            }

                            $timeRange = ! empty($data['hours']) ? $data['hours'] : 'All day';

                            return new HtmlString(
                                "<div class='flex justify-between p-2'>
                                    <div class='font-medium text-gray-500'>
                                        {$data['date_key']} - {$timeRange}
                                    </div>
                                    <div class='font-medium text-gray-500'>
                                        {$data['data']}
                                    </div>
                                </div>"
                            );
                        }),
                    Hidden::make('exception'),
                ])
                // ->visible(function (Get $get, $state) {
                //    return $state;
                // })
                ->columnSpan('full'),
        ]);

        $this->columns(1);

        $this->registerActions([
            Action::make('setupExceptions')
                ->label(__('filament-business-hours::business-hours.form.exceptions.action.label'))
                ->button()
                ->modalHeading('Add Exception')
                ->modalSubmitActionLabel(__('filament-business-hours::business-hours.form.exceptions.action.submit'))
                ->schema([
                    TextInput::make('date_key')
                        ->label(__('filament-business-hours::business-hours.form.exceptions.date.label'))
                        ->helperText(__('filament-business-hours::business-hours.form.exceptions.date.helper_text'))
                        ->required(),

                    Group::make()
                        ->schema([
                            TimePicker::make('start')
                                ->label(__('filament-business-hours::business-hours.form.exceptions.start.label'))
                                ->seconds(false)
                                ->helperText(__('filament-business-hours::business-hours.form.exceptions.start.helper_text')),
                            TimePicker::make('end')
                                ->label(__('filament-business-hours::business-hours.form.exceptions.end.label'))
                                ->seconds(false)
                                ->helperText(__('filament-business-hours::business-hours.form.exceptions.end.helper_text')),
                        ])
                        ->columns(2),

                    TextInput::make('description')
                        ->label(__('filament-business-hours::business-hours.form.exceptions.description.label'))
                        ->helperText(__('filament-business-hours::business-hours.form.exceptions.description.helper_text')),
                ])
                ->action(function (array $data, Set $set, Get $get) {
                    $key = $data['date_key'];

                    $hours = ($data['start'] && $data['end'])
                        ? $data['start'] . '-' . $data['end']
                        : '';

                    $exceptionData = [
                        'date_key' => $key,
                        'hours' => $hours,
                        'data' => $data['description'],
                    ];

                    $current = $this->getState();

                    $currentExceptions = $current['exceptions'];

                    // Add the new exception with proper repeater item structure
                    $currentExceptions[(string) Str::uuid()] = [
                        'exception' => json_encode($exceptionData),
                        'exception_placeholder' => json_encode($exceptionData),
                    ];

                    $set('exceptions', $currentExceptions);

                    Arr::set($current, 'exceptions', $currentExceptions);

                    $this->state([
                        ...$current,
                    ]);
                })
                ->color('gray'),
        ]);

        $this->afterStateHydrated(function (BusinessHoursField $component, $state) {
            // If we have a record and it has the businessHours relationship
            $record = $component->getRecord();

            if ($record) {
                $record->load('businessHours');

                if (isset($record->businessHours)) {
                    $relationState = $record->businessHours->toArray();
                    $formattedState = $this->transformBusinessHoursForForm($relationState);
                    $component->state($formattedState);

                    return;
                }
            }

            // Otherwise use default state
            $component->state([
                'enabled' => ! $component->getDisabledByDefault(),
                'timezone' => config('filament-business-hours.default_timezone'),
                'hours' => [],
                'exceptions' => [],
            ]);
        });

        $this->beforeStateDehydrated(function (BusinessHoursField $component, $state) {
            if (! $state) {
                return;
            }

            // Transform the data into the correct format before saving
            $transformed = $this->transformBusinessHours($state);

            $component->state($transformed);
        });

        $this->saveRelationshipsUsing(function (BusinessHoursField $component, ?array $state, Model $record) {
            if (! $state) {
                return;
            }

            // Ensure the model has the required businessHours method
            if (! method_exists($record, 'businessHours')) {
                throw new \InvalidArgumentException(
                    'Model ' . get_class($record) . ' must use the HasBusinessHours trait to work with BusinessHoursField.'
                );
            }

            try {
                // Get or create the business hours model
                $businessHours = $record->businessHours ?? $record->businessHours()->make();

                $businessHours->fill([
                    'hours' => $state['hours'] ?? [],
                    'exceptions' => $state['exceptions'] ?? [],
                    'timezone' => $state['timezone'] ?? config('filament-business-hours.default_timezone'),
                    'enabled' => $state['enabled'] ?? false,
                ]);

                // Save the relationship
                $record->businessHours()->save($businessHours);

                // Clear the cache if using caching
                if (method_exists($record, 'clearOpeningHoursCache')) {
                    $record->clearOpeningHoursCache();
                }

                // Transform the state back to form format after saving
                $formattedState = $this->transformBusinessHoursForForm([
                    'hours' => $state['hours'] ?? [],
                    'exceptions' => $state['exceptions'] ?? [],
                    'timezone' => $state['timezone'] ?? config('filament-business-hours.default_timezone'),
                    'enabled' => $state['enabled'] ?? false,
                ]);

                // Update the component state with the formatted data
                $component->state($formattedState);
            } catch (Exception $e) {
                throw new Exception("Failed to save business hours: {$e->getMessage()}");
            }
        });
    }

    protected function transformBusinessHours(array $state): array
    {
        // Transform the hours data
        $hours = collect($state)
            ->filter(fn ($_val, $key) => ! str_ends_with($key, '_enabled') && ! in_array($key, ['exceptions', 'timezone', 'enabled']))
            ->mapWithKeys(function ($ranges, $day) use ($state) {
                $enabled = $state["{$day}_enabled"] ?? false;

                if (! $enabled || empty($ranges)) {
                    return [$day => []];
                }

                $formatted = collect($ranges)
                    ->map(function ($pair) {
                        // Validate time format
                        if (! isset($pair['open']) || ! isset($pair['close'])) {
                            return null;
                        }

                        return "{$pair['open']}-{$pair['close']}";
                    })
                    ->filter()
                    ->values()
                    ->toArray();

                return [$day => $formatted];
            })
            ->filter()
            ->toArray();

        // Transform exceptions to Spatie format
        $exceptions = [];
        $formExceptions = $state['exceptions'] ?? [];

        foreach ($formExceptions as $exceptionData) {
            if (empty($exceptionData['exception'])) {
                continue;
            }

            $data = json_decode($exceptionData['exception'], true);
            if (! $data) {
                continue;
            }

            $dateKey = trim($data['date_key']);

            // Store the exception with the original date key (either single date or range)
            $exceptions[$dateKey] = [
                'hours' => ! empty($data['hours']) ? [$data['hours']] : [],
                'data' => $data['data'],
            ];
        }

        // Return the transformed state
        return [
            'hours' => $hours,
            'exceptions' => $exceptions,
            'timezone' => $state['timezone'] ?? config('filament-business-hours.default_timezone'),
            'enabled' => $state['enabled'] ?? false,
        ];
    }

    protected function transformBusinessHoursForForm(array $state): array
    {
        $formState = [
            'enabled' => $state['enabled'] ?? false,
            'timezone' => $state['timezone'] ?? config('filament-business-hours.default_timezone'),
        ];

        // Transform hours back into form format
        $hours = $state['hours'] ?? [];
        foreach ($this->getDays() as $day) {
            $formState["{$day}_enabled"] = ! empty($hours[$day] ?? []);

            if (! empty($hours[$day])) {
                $formState[$day] = collect($hours[$day])
                    ->map(function ($timeRange) {
                        [$open, $close] = explode('-', $timeRange);

                        return [
                            'open' => $open,
                            'close' => $close,
                        ];
                    })
                    ->toArray();
            } else {
                $formState[$day] = [];
            }
        }

        // Transform exceptions back to form format
        $exceptions = [];
        $spatieExceptions = $state['exceptions'] ?? [];

        foreach ($spatieExceptions as $date => $exceptionConfig) {
            $hours = $exceptionConfig['hours'] ?? [];

            $exceptionData = [
                'date_key' => $date,
                'hours' => ! empty($hours) ? $hours[0] : '', // Take first hours range
                'data' => $exceptionConfig['data'] ?? '',
            ];

            $exceptions[] = [
                'exception' => json_encode($exceptionData),
                'exception_placeholder' => json_encode($exceptionData),
            ];
        }

        $formState['exceptions'] = $exceptions;

        return $formState;
    }

    protected static function getDays(): array
    {
        return Days::values();
    }

    protected static function getDaySchema(string $day): Grid
    {
        return Grid::make([
            'default' => 4,
        ])
            ->schema([
                Toggle::make("{$day}_enabled")
                    ->label(__("filament-business-hours::business-hours.days.{$day}"))
                    ->live(),
                TextEntry::make("{$day}_closed")
                    ->hiddenLabel()
                    ->visible(function (Get $get) use ($day) {
                        return ! $get("{$day}_enabled");
                    })
                    ->state(view('filament-business-hours::forms.partials.closed')),
                Repeater::make($day)
                    ->extraAttributes(['class' => 'business-hours-table-repeater'])
                    ->hiddenLabel()
                    ->defaultItems(1)
                    ->reorderable(false)
                    ->table([
                        TableColumn::make('open')
                            ->hiddenHeaderLabel()
                            ->width('150px'),
                        TableColumn::make('close')
                            ->hiddenHeaderLabel()
                            ->width('150px'),
                    ])
                    ->schema([
                        TimePicker::make('open')
                            ->label(__('filament-business-hours::business-hours.from'))
                            ->prefix(__('filament-business-hours::business-hours.from'))
                            ->seconds(false),
                        TimePicker::make('close')
                            ->label(__('filament-business-hours::business-hours.to'))
                            ->prefix(__('filament-business-hours::business-hours.to'))
                            ->seconds(false)
                            ->extraFieldWrapperAttributes(['class' => 'pb-2']),
                    ])
                    ->addActionLabel(__('filament-business-hours::business-hours.add_' . $day . '_time'))
                    ->columnSpan('3')
                    ->visible(function (Get $get) use ($day) {
                        return $get("{$day}_enabled");
                    }),
            ]);
    }

    public function allowExceptions(bool | Closure $allowExceptions = true): static
    {
        $this->allowExceptions = $allowExceptions;

        return $this;
    }

    public function getAllowExceptions(): bool
    {
        return (bool) ($this->evaluate($this->allowExceptions) ?? false);
    }

    public function disabledByDefault(bool | Closure $disabledByDefault = true): static
    {
        $this->disabledByDefault = $disabledByDefault;

        return $this;
    }

    public function getDisabledByDefault(): bool
    {
        return (bool) ($this->evaluate($this->disabledByDefault) ?? false);
    }
}
