<?php

namespace Andreia\FilamentBusinessHours\Concerns;

use Andreia\FilamentBusinessHours\Models\BusinessHours;
use DateTimeInterface;
use Illuminate\Support\Facades\Cache;
use Spatie\OpeningHours\Exceptions\InvalidOpeningHours;
use Spatie\OpeningHours\OpeningHours;
use Spatie\OpeningHours\TimeRange;

trait HasBusinessHours
{
    /**
     * Relationship assume morphOne by default.
     */
    public function businessHours()
    {
        return $this->morphOne(BusinessHours::class, 'owner');
    }

    public function getOpeningHours(?bool $useCache = null): ?OpeningHours
    {
        $useCache = $useCache ?? config('business-hours.cache_enabled', true);

        if (! $useCache) {
            return $this->buildOpeningHours();
        }

        $cacheKey = "opening_hours:{$this->getMorphClass()}:{$this->getKey()}";

        return Cache::rememberForever($cacheKey, fn () => $this->buildOpeningHours());
    }

    protected function buildOpeningHours(): ?OpeningHours
    {
        if (! $this->businessHours || ! $this->businessHours->enabled) {
            return null;
        }

        try {
            return OpeningHours::create(
                array_merge(
                    $this->businessHours->hours ?? [],
                    [
                        'exceptions' => $this->businessHours->exceptions ?? [],
                        'overflow' => true,
                    ]
                ),
                $this->timezone,
            );
        } catch (InvalidOpeningHours $e) {
            report($e);

            return null;
        }
    }

    public function clearOpeningHoursCache(): void
    {
        Cache::forget("opening_hours:{$this->getMorphClass()}:{$this->getKey()}");
    }

    // Clear the cache any time this model is saved or deleted, to keep the OpeningHours instance in sync
    protected static function bootedHasBusinessHours(): void
    {
        static::saved(fn ($model) => $model->clearOpeningHoursCache());
        static::deleted(fn ($model) => $model->clearOpeningHoursCache());
    }

    /**
     * Check if the business is currently open.
     */
    public function isOpen(?DateTimeInterface $at = null): bool
    {
        $openingHours = $this->getOpeningHours();

        if (! $openingHours) {
            return false;
        }

        return $openingHours->isOpenAt($at ?? now());
    }

    /**
     * Get current open range or null.
     */
    public function currentOpenRange(?DateTimeInterface $at = null): ?TimeRange
    {
        return $this->getOpeningHours()?->currentOpenRange($at ?? now());
    }

    /**
     * Get next open time.
     */
    public function nextOpen(?DateTimeInterface $from = null): ?DateTimeInterface
    {
        return $this->getOpeningHours()?->nextOpen($from ?? now());
    }

    /**
     * Get next close time.
     */
    public function nextClose(?DateTimeInterface $from = null): ?DateTimeInterface
    {
        return $this->getOpeningHours()?->nextClose($from ?? now());
    }

    /**
     * Check if the business is currently closed.
     */
    public function isClosed(?DateTimeInterface $at = null): bool
    {
        return ! $this->isOpen($at);
    }
}
